/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       fileview.c
**     SYSTEM   NAME:       Beholder
**     ORIGINAL AUTHOR(S):  Ling Thio
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1991/9/2
**
** DESCRIPTION: Dumpfile viewing application for the online BEHOLDER
**
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.0  $
** WORKFILE:    $Workfile:   FILEVIEW.C  $
** LOGINFO:     $Log:   I:TIRZA/CPROG/BEHOLDER/BAPS/DPAPC/VCS/FILEVIEW.C_V  $
**              
**                 Rev 1.2   30 Aug 1991 12:44:12   tirza
**              Converted to new style function declarations, included
**              NVDGRAM.C, NVDISP.C, NVSEGM.C, NVUTI.C and NVVIEW.C , added
**              frame number to main screen and made it a BEHOLDER application
**
**                 Rev 1.1   12 Nov 1990 18:24:26   etstjan
**              No explicit note
**              
**                 Rev 1.0   24 Oct 1990 09:48:34   etstjan
**              Initial revision.
*************************************************************************/
#if !defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:TIRZA/CPROG/BEHOLDER/BAPS/DPAPC/VCS/PACVIEW.C_V   1.0   2 Sep 1991 18:24:26   tirza  $";
#endif


#include <stdlib.h>                         /* for _MAX_* */
#include <string.h>                         /* for strlen() */
#include <direct.h>                         /* for getcwd() */
#include <stdio.h>
#include <power3.h>
#include <error.h>
#include <dnpap.h>
#include <dp.h>
#include <dpu.h>


#define REPLACE         1                   /* modes for EthDisplayField() */
#define EXPLAIN         2
#define MAX_FRAMES      1000                /* maximum frames for view */
#define DEFAULT_DUMPFILE_NAME       "pktcapt.dmp" 


/*Function prototypes */
static int  ProcessEvent(DPEVENT Event);
static void Init(void);
static void End(void);
static int  Start(void);
static void Stop(void);
static void Reset(void);
static void Show(void);
static int PwmFileViewMenu(int element);
static int NvView(char *dumpfile);
static int proc_view(PWWIN *pWin, int action, int element);
static void NvDisplayHdrFields(PWWIN *win, DPBUF *frp, unsigned int framenr);
static void NvDisplayExtra(PWWIN *win, DPBUF *frp);
static void DgramDisplay(DPBUF *frp);
static void NvDisplayFieldDef(PWWIN *win, char *fname, BYTE *key, int mode);
static void NvDisplayHex(PWWIN *win, BYTE *buf, unsigned len);
static void DgramIPX_Nov(DPBUF *frp);
static void DgramIP(DPBUF *frp);
static void DgramARP(DPBUF *frp);
static void DgramDEC(DPBUF *frp);
static void DgramUnknown(DPBUF *frp);
static void NvDisplayLen(PWWIN *win, unsigned len);
static int SegmICMP(PWWIN *segmwin, BYTE *buf);
static int SegmST(PWWIN *segmwin, BYTE *buf);
static int SegmTCP(PWWIN *segmwin, BYTE *buf);
static int SegmUDP(PWWIN *segmwin, BYTE *buf);
static PWKEY NvDisplayHexDump(PWWIN *win, BYTE *buf, int offset, int len);
static void NvDisplayIpAddress(PWWIN *win, BYTE *addr);
static void display_hexline(PWWIN *win, int line, BYTE *buf, int len);


DPAPPS DpaFileView = {
    "FileView",                             /* application name */
    ProcessEvent,                           /* event processing function */
    0,                                      /* status bits, must be 0 */
/* accept event mask, tells the beholder which events this */
/* application wishes to handle                              */
/*   DPE_DEFAULT = DPE_INIT | DPE_END | DPE_START | DPE_STOP */
    DPE_DEFAULT,
/* start event mask, tells the beholder which events to          */
/* generate for this application during the 'DpStart()' function  */
    DPE_RESET | DPE_SHOW,
    0,                                           /* timer event mask */
    0,                                           /* timer value */
    0,                                           /* timer start */
    0,                                           /* timer stop */
    0,                                           /* timer timer */
    0,                                           /* # of active filters */
    0,                                           /* active filter # */
    {0}                                          /* list of active filters */
};
static long fpos[MAX_FRAMES];
static DPBUF gfr;
static BYTE gbuf[1514];
static unsigned int nframes;
static char *viewfile;
static char TimeStamp[18] = "";
static char format[6];
static char scratch[81];
static char ViewfileName[80];
static char *FileViewMenu[] = {
    "View",
    "Dumpfile",
    NULL
};
static int FileViewElem;
static int FileElem = 0;


static int ProcessEvent(DPEVENT Event)
{
    int ret = 0;

    switch(Event)
    {
        case DPE_INIT:          Init();         break;
        case DPE_END:           End();          break; 
        case DPE_START:   ret = Start();        break;
        case DPE_STOP:          Stop();         break;
        case DPE_SHOW:          Show();         break;
        case DPE_RESET:         Reset();        break;
    }
    return ret;
}

static void Init(void)
{
}

static void End(void)
{
}

static int Start(void)
{
    if (DpuIniInit("DP.INI") == 0)
    {
        strcpy (ViewfileName, DEFAULT_DUMPFILE_NAME); 
        pw_menu(10, 14, 0, 0, "File view", FileViewMenu, PwmFileViewMenu, &FileViewElem);
        DpAppsEvent(DpAppsGetApps("FileView"), DPE_STOP);
        return (0);
    }
    else
    {
        pw_error("Can't open init file DP.INI");
        return (1);
    }        
}

static void Stop(void)
{
}

static void Reset(void)
{
}

static void Show(void)
{
}

static int PwmFileViewMenu(int element)
{
    switch (element)
    {
        case 0: /* View */
            if (NvView(ViewfileName) != 0)
                pw_error("Can't open data file '%s'", ViewfileName);
            break;
        case 1: /* Dumpfile */
            DpuGetFile("Dumpfile", "*.DMP", ViewfileName, &FileElem);
            break;
    }
    return element;
}

static int NvView(char *dumpfile)
{
    int ret = 0;
    int element = 0;

    if ((ret = DpuDumpInit(dumpfile, FR_DUMP_READ)) == 0)
    {
        pw_waiton();
        nframes = DpuDumpCount(fpos, MAX_FRAMES, TimeStamp);
        pw_waitoff();
        viewfile = dumpfile;
        pw_select(2, 1, 21, 78, "Frame", proc_view, &element);
        DpuDumpEnd();
    }
    return ret;
}

static int proc_view(PWWIN *pWin, int action, int element)
{
    char buffer[_MAX_PATH];

    switch (action)
    {
        case PWA_DISPLAY:
            DpuDumpSeek(fpos[element]);
            DpuDumpReadFrame(&gfr, gbuf);
            pw_printf(pWin, "%05d ", element+1);
            NvDisplayHdrFields(pWin, &gfr, element);
            NvDisplayExtra(pWin, &gfr);
            break;
        case PWA_ENTER:
            DpuDumpSeek(fpos[element]);
            DpuDumpReadFrame(&gfr, gbuf);
            DgramDisplay(&gfr);
            break;
        case PWA_INSERT:
            break;
        case PWA_DELETE:
            break;
        case PWA_COUNT:
            return nframes;
            break;
        case PWA_FIRSTCHAR:
            break;
        case PWA_INIT:
            pw_putsat(pWin, -1, 16, "S");
            pw_putsat(pWin, -1, 19, "Source");
            pw_putsat(pWin, -1, 33, "Dest");
            pw_putsat(pWin, -1, 47, "Type");
            pw_putsat(pWin, -1, 56, "Len");
            pw_putsat(pWin, -1, 60, "Info"); 
            pw_putsat(pWin, pWin->nrow, 0, "");
            if (!strchr(viewfile, '\\') && !strchr(viewfile, ':'))
            {
                if (getcwd(buffer, _MAX_PATH-1))
                {
                    pw_putsat(pWin, pWin->nrow, 0, buffer);
                    pw_putc(pWin, '\\');
                }
            }
            pw_puts(pWin, viewfile);
            pw_putsat(pWin, pWin->nrow, pWin->ncol-17, TimeStamp);
            break;
    }
    return element;
}

static void NvDisplayHdrFields(PWWIN *win, DPBUF *frp, unsigned int framenr)
/*****************************************************************
 * Display a frame header: "SRC-->DEST, T=Type, L=Length"        *
 *****************************************************************/
{
    int s, ms, us;

    us = (int)(frp->ClockMs % 1000);
    ms = (int)((frp->ClockMs / 1000L) % 1000);
    s  = (int)((frp->ClockMs / 1000000L) % 1000);
    if (framenr != 0xffff)
        pw_printf(win, "%3d:%03d:%03d ", s, ms, us);
    NvDisplayFieldDef(win, "HDRADDR", frp->pBuf+hdrinfo->SrcOff, REPLACE);
    pw_puts(win, " ");
    NvDisplayFieldDef(win, "HDRADDR", frp->pBuf+hdrinfo->DestOff, REPLACE);
    pw_puts(win, " ");
    NvDisplayFieldDef(win, "HDRTYPE", frp->pBuf+hdrinfo->TypOff, REPLACE);
    pw_printf(win, " %4d", frp->Size);
}

static void NvDisplayExtra(PWWIN *win, DPBUF *frp)
{
    unsigned int type = (frp->pBuf[hdrinfo->TypOff] << 8) + frp->pBuf[hdrinfo->TypOff+1];

    pw_putc(win, ' ');
    if (type < 0x0600)
    {
        switch (frp->pBuf[hdrinfo->DatOff])
        {
            case 0xff:
                pw_puts(win, "IPX_Nov");
                break;
            case 0xf0:
                pw_puts(win, "IPX_Lan");
                break;
            default:
                pw_puts(win, "802.3");
                break;
        }
    }
    else
    {
        switch (type)
        {
            case 0x0800:                            /* IP */
                NvDisplayFieldDef(win, "IPPROT", frp->pBuf+hdrinfo->DatOff+9, REPLACE);
                break;
            case 0x0806:                            /* ARP datagram */
            case 0x8035:                            /* RARP datagram */
                pw_puts(win, "(R)ARP");
                break;
/*            NvDisplayFieldDef(win, "DECTYPE", frp->pBuf+hdrinfo->DatOff+0x16, REPLACE);
            break; */
            case 0x6007:
            case 0x6003:                            /* DEC datagream */
            case 0x6004:
                pw_puts(win, "DECNET");
                break;
        }
    }
}

static void DgramDisplay(DPBUF *frp)
{
    unsigned int type = (frp->pBuf[hdrinfo->TypOff] << 8) + frp->pBuf[hdrinfo->TypOff+1];

    if (type < 0x0600)
    {
        switch (frp->pBuf[hdrinfo->DatOff])
        {
            case 0xff:
                DgramIPX_Nov(frp);              /* IPX Novell datagram */
                break;
            case 0xf0:
            default:
                DgramUnknown(frp);              /* Unknown datagram */
                break;
        }
    }
    else
    {
        switch (type)
        {
            case 0x0800:                       
                DgramIP(frp);                   /* IP datagram */
                break;
            case 0x0806:                        /* ARP datagram */
            case 0x8035:                        /* RARP datagram */
                DgramARP(frp);
                break;
            case 0x6003:                        
            case 0x6004:
            case 0x6007:
                DgramDEC(frp);                  /* DEC datagram */
                break;
            default:
                DgramUnknown(frp);              /* Unknown datagram */
                break;
        }
    }
}

static void NvDisplayFieldDef(PWWIN *win, char *fname, BYTE *key, int mode)
/*****************************************************************
 * Display a BYTE string, or its description (when defined).     *
 *****************************************************************/
{
    FIELDDEF *ptr = NULL;
    long l;
    int i;
    int id = DpuIniFindId(fname);

    if (id >= 0)
        ptr = DpuIniFindDef(id, key);
    if ((!ptr) || (mode != REPLACE))
    {
        if (fieldinfo[id].hex)
        {
            NvDisplayHex(win, key, fieldinfo[id].keylen);
            pw_putc(win, 'h');
            if (mode == REPLACE)
                for (i=fieldinfo[id].keylen*2+1; i<fieldinfo[id].desclen; i++)
                    pw_putc(win, ' ');
        }
        else
        {
            l = 0;
            for (i=0; i<fieldinfo[id].keylen; i++)
                l = l * 256 + (long)key[i];
            sprintf(format, "X%dld", fieldinfo[id].keylen*3);
            *format = '%';
            pw_printf(win, format, l);
        }
    }
    sprintf(format, "X-%ds", fieldinfo[id].desclen);
    *format = '%';
    if (ptr)
        sprintf(scratch, format, ptr->desc);
    else
    {
        sprintf(scratch, format, "Unknown");
        scratch[fieldinfo[id].desclen] = '\0';
    }
    if (mode != REPLACE)
    {
        pw_puts(win, " (");
        pw_puts(win, scratch);
        pw_putc(win, ')');
    }
    else
    {
        if (ptr)
            pw_puts(win, scratch);
    }
}

static void NvDisplayHex(PWWIN *win, BYTE *buf, unsigned len)
/*****************************************************************
 * Display a 'BYTE' string in hexadecimal format                 *
 *****************************************************************/
{
    int i;

    for (i=0; i< (int)len; i++)
        pw_printf(win, "%02x", buf[i]);
}

static void DgramIPX_Nov(DPBUF *frp)
{
    PWWIN *dgramwin;
    PWWIN *datawin;
    BYTE *buf = frp->pBuf+hdrinfo->DatOff;
    unsigned int len = (buf[2] << 8) + buf[3];

    dgramwin = pw_open(1, 13, 5, 66, "IPX datagram header", 0, PWW_HELP);
    NvDisplayLen(dgramwin, 30);
    pw_putsat(dgramwin, 0, 0, "Frame: ");
    NvDisplayHdrFields(dgramwin, frp, -1);
    pw_printf(dgramwin, "\nDgram: Chksum=%02x%02xh, Len =%4d", buf[0], buf[1], len);
    sprintf(scratch, "\n       TrCtr =  %02xh, Type= ", buf[4]);
    pw_puts(dgramwin, scratch);
    NvDisplayFieldDef(dgramwin, "IPXTYPE", buf+5, EXPLAIN);
    pw_puts(dgramwin, "\n       Dest:   Netw=");
    NvDisplayHex(dgramwin, buf+6, 4);
    pw_puts(dgramwin, ", Node=");
    NvDisplayFieldDef(dgramwin, "HDRADDR", buf+10, REPLACE);
    pw_puts(dgramwin, ", Sock=");
    NvDisplayHex(dgramwin, buf+16, 2);
    if ((buf[16] == 0x04) && (buf[17] == 0x51))
        pw_puts(dgramwin, " (FS)");
    pw_puts(dgramwin, "\n       Source: Netw=");
    NvDisplayHex(dgramwin, buf+18, 4);
    pw_puts(dgramwin, ", Node=");
    NvDisplayFieldDef(dgramwin, "HDRADDR", buf+22, REPLACE);
    pw_puts(dgramwin, ", Sock=");
    NvDisplayHex(dgramwin, buf+28, 2);
    if ((buf[28] == 0x04) && (buf[29] == 0x51))
        pw_puts(dgramwin, " (FS)");
    datawin = pw_open(11, 13, 12, 66, "Data", 0, PWW_HELP);
    NvDisplayLen(datawin, len-30);
    pw_keyinfo(datawin, NULL, NULL);
    if (NvDisplayHexDump(datawin, buf, 30, len) == 'R' - '@')
        NvDisplayHexDump(datawin, buf, 0, len);
    pw_close(datawin);
    pw_close(dgramwin);
}

static void DgramIP(DPBUF *frp)
{
    PWWIN *dgramwin;
    PWWIN *datawin;
    PWWIN *segmwin = (PWWIN *)0;
    BYTE *buf = frp->pBuf+hdrinfo->DatOff;
    int offset;
    int dgramoffset = (buf[0] & 0x0f) * 4;
    int i;
    unsigned int tmp;
    unsigned int len = (buf[2] << 8)+buf[3];
    unsigned int offs = ((buf[6] & 0x0f) << 8) + buf[7];

    dgramwin = pw_open(1, 13, 5, 66, "IP datagram header", 0, PWW_HELP);
    NvDisplayLen(dgramwin, dgramoffset);
    pw_putsat(dgramwin, 0, 0, "Frame: ");
    NvDisplayHdrFields(dgramwin, frp, 0xffff);
    pw_printf(dgramwin, "\nDgram: Ver= %2d , IHL=  %2d , ToS=",
        (buf[0] >> 4), (buf[0] & 0x0f));
    tmp = buf[1];
    for (i=0; i<8; i++)
    {
        pw_putc(dgramwin, (tmp & 0x80) ? '1': '0');
        tmp = (tmp << 1);
    }
    pw_printf(dgramwin, "b, Len= %4d", len);
    pw_printf(dgramwin, "\n       Id=%04xh,            Flags=  ", ((buf[4] << 8)+buf[5]));
    tmp = buf[6];
    for (i=0; i<4; i++)
    {
        pw_putc(dgramwin, (tmp & 0x80) ? '1': '0');
        tmp = (tmp << 1);
    }
    pw_printf(dgramwin, "b, Offs=%4d", offs);
    pw_printf(dgramwin, "\n       TtL=%3d , Prot=%3d , HChksum=%04xh",
        buf[8], buf[9], ((buf[10] << 8) + buf[11]));
    pw_printf(dgramwin, "\n       IPSrc=%d.%d.%d.%d, IPDest=%d.%d.%d.%d",
        buf[12], buf[13], buf[14], buf[15],
        buf[16], buf[17], buf[18], buf[19]);
    offset = dgramoffset;
    switch (buf[9])
    {
        case 1:
            if (!offs)
            {
                segmwin = pw_open(7, 13, 3, 66, "ICMP segment header", 0, PWW_HELP);
                offset = dgramoffset + SegmICMP(segmwin, buf+dgramoffset);
            }
            break;
        case 5:
            if (!offs)
            {
                segmwin = pw_open(7, 13, 3, 66, "ST segment header", 0, PWW_HELP);
                offset = dgramoffset + SegmST(segmwin, buf+dgramoffset);
            }
            break;
        case 6:
            if (!offs)
            {
                segmwin = pw_open(7, 13, 3, 66, "TCP segment header", 0, PWW_HELP);
                offset = dgramoffset + SegmTCP(segmwin, buf+dgramoffset);
                /* 00 padding so cnt >= 60, so actual length may be less */
                frp->Size = ((buf[2] << 8)+buf[3]) + hdrinfo->DatOff;
            }
            break;
        case 17:
            if (!offs)
            {
                segmwin = pw_open(7, 13, 3, 66, "UDP segment header", 0, PWW_HELP);
                offset = dgramoffset + SegmUDP(segmwin, buf+dgramoffset);
            }
            break;
    }
    datawin = pw_open(11, 13, 12, 66, "Data", 0, PWW_HELP);
    NvDisplayLen(datawin, len-offset);
    pw_keyinfo(datawin, NULL, NULL);
    if (NvDisplayHexDump(datawin, buf, offset, len) == 'R'-'@')
    {
        NvDisplayHexDump(datawin, buf, 0, len);
        pw_putcat(datawin, (dgramoffset & 0x0f) * 3, (dgramoffset >> 4), '*');
        pw_putcat(datawin, (offset & 0x0f) * 3, (offset >> 4), '*');
    }
    pw_close(datawin);
    if (segmwin)
        pw_close(segmwin);
    pw_close(dgramwin);
}

static void DgramARP(DPBUF *frp)
{
    PWWIN *dgramwin;
    BYTE *buf = frp->pBuf+hdrinfo->DatOff;
    int hlen = buf[4];
    int plen = buf[5];

    dgramwin = pw_open(1, 13, 6, 66, "(R)ARP datagram header", 0, PWW_HELP);
    NvDisplayLen(dgramwin, 6+hlen*2+plen*2);
    pw_keyinfo(dgramwin, NULL, NULL);
    pw_putsat(dgramwin, 0, 0, "Frame: ");
    NvDisplayHdrFields(dgramwin, frp, -1);
    pw_puts(dgramwin, "\nHardware Address Space: ");
    NvDisplayHex(dgramwin, buf, 2);
    pw_printf(dgramwin, "h, Length: %02xh", hlen);
    pw_puts(dgramwin, "\nProtocol Address Space: ");
    NvDisplayHex(dgramwin, buf+2, 2);
    pw_printf(dgramwin, "h, Length: %02xh", plen);
    pw_puts(dgramwin, "\nOpcode = ");
    NvDisplayHex(dgramwin, buf+6, 2);
    pw_puts(dgramwin, "h  ");
    switch((buf[6]<<8) + buf[7])
    {
        case 1:
            pw_puts(dgramwin, "(Request)");
            break;
        case 2:
            pw_puts(dgramwin, "(Reply)");
            break;
        case 3:
            pw_puts(dgramwin, "(Request reverse)");
            break;
        case 4:
            pw_puts(dgramwin, "(Reply reverse)");
            break;
        default:
            pw_puts(dgramwin, "(Unknown)");
            break;
        
    }
    pw_puts(dgramwin, "\nSource:  Hardware: ");
    NvDisplayHex(dgramwin, buf+8, hlen);
    pw_puts(dgramwin, "h, Protocol: ");
    NvDisplayIpAddress(dgramwin, buf+8+hlen);
    pw_puts(dgramwin, "\nDest:    Hardware: ");
    NvDisplayHex(dgramwin, buf+8+hlen+plen, hlen);
    pw_puts(dgramwin, "h, Protocol: ");
    NvDisplayIpAddress(dgramwin, buf+8+hlen+plen+hlen);
    pw_getkey();

    pw_close(dgramwin);
}

static void DgramDEC(DPBUF *frp)
{
    PWWIN *dgramwin;
    PWWIN *datawin;
    BYTE *buf = frp->pBuf+hdrinfo->DatOff;
    unsigned int len = *((unsigned int *)buf);

    dgramwin = pw_open(1, 13, 5, 66, "DEC datagram header", 0, PWW_HELP);
    NvDisplayLen(dgramwin, 0x17);
    pw_putsat(dgramwin, 0, 0, "Frame: ");
    NvDisplayHdrFields(dgramwin, frp, -1);
    pw_printf(dgramwin, "\nLength   =%4d", len);
    pw_puts(dgramwin,   "\nDX: Dest =");
    NvDisplayHex(dgramwin, buf+2, 6);
    pw_puts(dgramwin,   "h, Group=");
    NvDisplayHex(dgramwin, buf+8, 2);
    pw_puts(dgramwin,   "h, Src=");
    NvDisplayHex(dgramwin, buf+0x0a, 6);
    pw_puts(dgramwin,   "h");
    pw_printf(dgramwin, "\nTR: Flags=%02xh, PAD=%3d,  ACK=", buf[0x10], buf[0x11]);
    NvDisplayHex(dgramwin, buf+0x12, 2);
    pw_puts(dgramwin,   "h,   SEQ=");
    NvDisplayHex(dgramwin, buf+0x14, 2);
    pw_puts(dgramwin,   "h");
    pw_puts(dgramwin,   "\nDECtype  =");
    NvDisplayFieldDef(dgramwin, "DECTYPE", frp->pBuf+hdrinfo->DatOff+0x16, EXPLAIN);
    datawin = pw_open(11, 13, 12, 66, "Data", 0, PWW_HELP);
    NvDisplayLen(datawin, len-0x17);
    pw_keyinfo(datawin, NULL, NULL);
    if (NvDisplayHexDump(datawin, buf, 0x17, len) == 'R'-'@')
        NvDisplayHexDump(datawin, buf, 0, len);
    pw_close(datawin);
    pw_close(dgramwin);
}

static void DgramUnknown(DPBUF *frp)
{
    PWWIN *dgramwin;
    PWWIN *datawin;

    dgramwin = pw_open(1, 13, 5, 66, "Unknown frame type", 0, PWW_HELP);
    pw_putsat(dgramwin, 0, 0, "Frame: ");
    NvDisplayHdrFields(dgramwin, frp, 0xffff);
    datawin = pw_open(11, 13, 12, 66, "Data", 0, PWW_HELP);
    NvDisplayLen(datawin, frp->Size-hdrinfo->DatOff);
    pw_keyinfo(datawin, NULL, NULL);
    NvDisplayHexDump(datawin, frp->pBuf+hdrinfo->DatOff, 0, frp->Size-hdrinfo->DatOff);
    pw_close(datawin);
    pw_close(dgramwin);
}

static void NvDisplayLen(PWWIN *win, unsigned len)
{
    char temp[7];

    sprintf(temp, "(%d)", len);
    pw_putsat(win, -1, win->ncol-strlen(temp), temp);
}

static int SegmICMP(PWWIN *segmwin, BYTE *buf)
{
    pw_puts(segmwin, "Type     =  ");
    NvDisplayFieldDef(segmwin, "ICMPTYPE", buf, EXPLAIN);
    pw_printf(segmwin, "\nCode   = %4d", buf[1]);
    pw_puts(segmwin, "\nChksum = ");
    NvDisplayHex(segmwin, buf+2, 2);
    pw_putc(segmwin, 'h');
    NvDisplayLen(segmwin, 4);
    return 4;                       /* header length in octets */
}

static int SegmST(PWWIN *segmwin, BYTE *buf)
{
    int headerlen = (buf[1] << 1);

    pw_printf(segmwin, " ST=%2d, Version=%2d, STHL=%3d Octets",
        (buf[0] >> 4), (buf[0] & 0x0f), headerlen);
    pw_printf(segmwin, "\n Total len=%5d Octets, Chksum=%02x%02x",
        ((buf[2] << 8) + buf[3]), buf[4], buf[5]);
    pw_putsat(segmwin,-1, segmwin->ncol-3, "(6)");
    return 6;                       /* header length in octets */
}

static int SegmTCP(PWWIN *segmwin, BYTE *buf)
{
    WORD port;
    BYTE flags, opt;
    unsigned long seq, ack;
    unsigned int window, urgp, mss;
    int headerlen = (buf[12] >> 4) * 4;
    int i, undef=0;

    port = (buf[0] << 8) + buf[1];        /* Source Port */
    pw_puts(segmwin, " Src Port=");                         
    NvDisplayFieldDef(segmwin, "TCPPORT", buf, EXPLAIN);
    port = (buf[2] << 8) + buf[3];        /* Destination Port */
    pw_puts(segmwin, " Dest Port=");
    NvDisplayFieldDef(segmwin, "TCPPORT", buf+2, EXPLAIN);
    seq = (buf[7] << 24) + (buf[6] << 16) + (buf[5] << 8) + buf [4];     /* Sequence Number */
    ack = (buf[11] << 24) + (buf[10] << 16) + (buf[9] << 8) + buf [8];   /* Acknowledgement Number */
    window = (buf[14] << 8) + buf[15];    /* Window */
    pw_printf(segmwin, "\n Seq=%10lu Ack=%10lu DataOff=%3d Window=%5u",
        seq, ack, headerlen, window);
    flags = buf[13];                      /* 'Flags' */
    urgp = (buf[18] << 8) + buf[19];      /* Urgent Pointer */
    pw_printf(segmwin, "\n URG=%1d ACK=%1d PSH=%1d RST=%1d SYN=%1d FIN=%1d UP=%9u",
        (flags>>5)&0x01, (flags>>4)&0x01, (flags>>3)&0x01,
        (flags>>2)&0x01, (flags>>1)&0x01, flags&0x01, urgp);
    if (headerlen > 20) /* Options field exists */
    {
        i=20;
        opt = buf[i];
        i++;
        while ((opt != 0x00) && (!undef))  /* End of Option List */
        {
            switch (opt)
            {
                case 0x01:   /* No-Operation */
                    break;
                case 0x02:
                    opt = buf[i];
                    i++;
                    if (opt == 0x04)   /* Maximum Segment Size */
                    {
                        mss = (buf[i] << 8) + buf[i+1];
                        i++;
                        pw_printf(segmwin, " MSS=%9u");
                    }
                    else
                    {
                        undef++;
                        pw_printf(segmwin, " Opt=Unknown");
                    }
                    break;
                default:    /* Undefined option */
                    undef++;
                    pw_printf(segmwin, " Opt=Unknown");
                    break;
            }
            opt = buf[i];
            i++;
        }
    }
    NvDisplayLen(segmwin, headerlen);
    return headerlen;                       /* header length in octets */
}

static int SegmUDP(PWWIN *segmwin, BYTE *buf)
{
    WORD port;

    port = (buf[0] << 8) + buf[1];
    pw_puts(segmwin, " Src Port=");
    NvDisplayFieldDef(segmwin, "TCPPORT", buf, EXPLAIN);
    port = (buf[2] << 8) + buf[3];
    pw_puts(segmwin, " Dest Port=");
    NvDisplayFieldDef(segmwin, "TCPPORT", buf+2, EXPLAIN);
    pw_printf(segmwin, "\n Length = %4d, Chksum = %02x%02xh",
        ((buf[4] << 8) + buf[5]), buf[6], buf[7]);
    NvDisplayLen(segmwin, 8);
    return 8;                               /* header length in octets */
}

static PWKEY NvDisplayHexDump(PWWIN *win, BYTE *buf, int offset, int len)
#ifdef HISTORY
{
    return pw_hexdump(11, 10, 12, buf+offset, len-offset);
}
#else
{
    int i;
    int line = 0;
    int nline = ((len-offset) >> 4) + 1;
    int topline = 0;
    int endloop=0;
    PWKEY key;

    if (nline > win->nrow)
        pw_keyinfo(win, "F1= Help  F9=Keys  Esc=Exit", "Cursor keys=Browse");
    else
        pw_keyinfo(win, "F1= Help  F9=Keys  Esc=Exit", "");
    while ((line < nline) && (line < win->nrow))
    {
        i = (line << 4) + offset;
        display_hexline(win, line, buf+i, len-i);
        line++;
    }
    while (!endloop)
    {
        switch (key = pw_getkey())
        {
            case PWK_UP:
                if (topline)
                {
                    topline--;
                    pw_vscroll(win, 0, 0, 0, 0, -1);
                    line = (topline << 4) + offset;
                    display_hexline(win, 0, buf+line, len-line);
                }
                break;
            case PWK_DOWN:
                if ((nline > win->nrow) && (topline < nline-win->nrow))
                {
                    topline++;
                    pw_vscroll(win, 0, 0, 0, 0, 1);
                    line = ((topline+win->nrow-1) << 4) + offset;
                    display_hexline(win, win->nrow-1, buf+line, len-line);
                }
                break;
            case PWK_PGUP:
                if (topline)
                {
                    topline = topline-win->nrow;
                    if (topline < 0)
                        topline = 0;
                    line = 0;
                    while ((line < nline-topline) && (line < win->nrow))
                    {
                        i = (topline+line << 4) + offset;
                        display_hexline(win, line, buf+i, len-i);
                        line++;
                    }
                }
                break;
            case PWK_PGDN:
                if (topline+win->nrow < nline)
                {
                    topline = topline + win->nrow;
                    pw_cls(win);
                    line = 0;
                    while ((line < nline-topline) && (line < win->nrow))
                    {
                        i = (topline+line << 4) + offset;
                        display_hexline(win, line, buf+i, len-i);
                        line++;
                    }
                }
                break;
            default:
                endloop++;
                break;
        }
    }
    return key;
}
#endif

static void NvDisplayIpAddress(PWWIN *win, BYTE *addr)
{
    pw_printf(win, "%d.%d.%d.%d", addr[0], addr[1], addr[2], addr[3]);
}

static void display_hexline(PWWIN *win, int line, BYTE *buf, int len)
/************************************************************************
 * Display a 16 'BYTE' string in 'debug dump' format                    *
 * "xx xx xx xx xx xx xx xx-xx xx xx xx xx xx xx xx  aaaaaaaa-aaaaaaaa" *
 ************************************************************************/
{
    char temp[17];
    int col = 0;
    int c;

    pw_putcat(win, line, 0, ' ');
    while ((col < len) && (col < 16))
    {
        c = buf[col];
        pw_printf(win, "%02x ", c);
        if ((c < ' ') || (c > ''))
            c = '';
        temp[col++] = (char)c;
    }
    temp[col] = (char)'\0';
    pw_putsat(win, line, 49, temp);
}
