/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       DPUDUMP.C
**     SYSTEM   NAME:       PACKET DISPATCHER
**     ORIGINAL AUTHOR(S):  Ling Thio
**     VERSION  NUMBER:     v1.00
**     CREATION DATE:       1989/11/30
**
** DESCRIPTION: Functions to read / write dumpfile
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   DPUDUMP.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/C600/BEHOLDER/NPD/DPU/VCS/DPUDUMP.C_V  $
**              
**                 Rev 1.1   25 Sep 1991 15:20:19   tirza
**              Changed and speeded up DpuDumpCount. Converted to new style
**              function declarations.
**              
**                 Rev 1.0   25 Oct 1990 15:26:54   etstjan
**              Initial revision.
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/C600/BEHOLDER/NPD/DPU/VCS/DPUDUMP.C_V   1.0   25 Oct 1990 15:26:54   etstjan  $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>                           /* for _strdate, _strtime */
#include "dpuinc.h"                         /* main include file */

#define REC_HEADER          0xfe            /* Dump file record types */
#define REC_VERSION         0xfd
#define REC_HDRINFO         0xfc
#define REC_TIMESTAMP       0xfb
#define REC_FRAME           0xf8
#define FR_DUMP_HDRSIZE     (1+sizeof(unsigned int)) /* ID + len */

extern DPHDRINFO *hdrinfo;

static char rec_header[] = "FRAME dumpfile";
static char rec_version[] = "v1.00";
static FILE *dumpfh;
static char *buf;

/************************************************************************
** NAME:        DpuDumpInit
** SYNOPSIS:    int DpuDumpInit(char *dumpfile, int mode)
**              dumpfile      Dump file filename
**              mode          Open mode
** DESCRIPTION: Opens dumpfile for read, create or append.
**                Modes are defined in DP.H:
**                  FR_DUMP_APPEND
**                  FR_DUMP_CREATE
**                  FR_DUMP_READ
** RETURNS:     0:  Successful call
**              -1: Can't open file
** SEE ALSO:    DumpEnd, DpuDumpReadFrame, DpuDumpWriteFrame
*************************************************************************/
int DpuDumpInit(char *dumpfile, int mode)
{
    unsigned int len;
    char temp[18];

    if (!(*dumpfile))
    {
        dumpfh = (FILE *)0;
        return 0;                           /* don't dump to file */
    }
    switch (mode)
    {
        case FR_DUMP_CREATE:
            dumpfh = fopen(dumpfile, "wb");
            break;
        case FR_DUMP_APPEND:
            dumpfh = fopen(dumpfile, "ab");
            break;
        default:
            dumpfh = fopen(dumpfile, "rb");
            break;
    }
    if (!dumpfh)
        return -1;                          /* Can't open dumpfile */
    if ((buf = (char*)malloc(16*1024*sizeof(char))) != NULL)
        setvbuf(dumpfh, buf, _IOFBF, 16*1024);
    fseek(dumpfh, 0L, SEEK_END);
    if (!ftell(dumpfh))
    {
        fputc(REC_HEADER, dumpfh);
        len = sizeof(rec_header) + FR_DUMP_HDRSIZE;
        fwrite(&len, sizeof(len), 1, dumpfh);
        fwrite(rec_header, sizeof(rec_header), 1, dumpfh);

        fputc(REC_VERSION, dumpfh);
        len = sizeof(rec_version) + FR_DUMP_HDRSIZE;
        fwrite(&len, sizeof(len), 1, dumpfh);
        fwrite(rec_version, sizeof(rec_version), 1, dumpfh);

        fputc(REC_HDRINFO, dumpfh);
        len = sizeof(DPHDRINFO) + FR_DUMP_HDRSIZE;
        fwrite(&len, sizeof(len), 1, dumpfh);
        fwrite(hdrinfo, sizeof(DPHDRINFO), 1, dumpfh);

        fputc(REC_TIMESTAMP, dumpfh);
        len = sizeof(temp) + FR_DUMP_HDRSIZE;
        _strdate(temp);
        temp[8] = ' ';
        _strtime(temp+9);
        fwrite(&len, sizeof(len), 1, dumpfh);
        fwrite(temp, sizeof(temp), 1, dumpfh);

        fflush(dumpfh);                     /* speeds up monitor start */
    }
    return 0;
}

/************************************************************************
** NAME:        DpuDumpCount
** SYNOPSIS:    unsigned int DpuDumpCount(long fpos[],
**                        unsigned int maxframes, char *TimeStamp)
**              fpos        Destination array for file positions of
**                          packet frames in dumpfile
**              maxframes   Maximum number of packet frames to be read
**              TimeStamp   Destination string for dumpfile timestamp 
** DESCRIPTION: Counts the number of packet frames in dumpfile, extracts
**              the dumpfile timestamp and determines the file positions
**              of the packet frames.
** RETURNS:     nframes:  number of packet frames read
** SEE ALSO:    
*************************************************************************/
unsigned int DpuDumpCount(long fpos[], unsigned int maxframes, char *TimeStamp)
{
    unsigned int nframes = 0;
    long fptr = 0L;
    int len;
    unsigned char rec;
    BYTE buffer[1600];

    if (dumpfh)
    {
        fseek(dumpfh, fptr, SEEK_SET);
        while ((fread(&rec, sizeof(unsigned char), 1, dumpfh) == 1)               &&
               (fread(&len, sizeof(int), 1, dumpfh) == 1)      &&
               ((int)fread(&buffer, sizeof(BYTE), (len-3), dumpfh) == (len-3)) &&
               (nframes < maxframes))
        {
            switch (rec)
            {
                case REC_TIMESTAMP:
                    memcpy(TimeStamp, buffer, len-FR_DUMP_HDRSIZE);
                    break;
                case REC_FRAME:
                    if (fpos)
                        fpos[nframes++] = fptr;
                    break;
            }
            fptr += len;
        }
    }
    return nframes;
}

/************************************************************************
** NAME:        DpuDumpSeek
** SYNOPSIS:    void DpuDumpSeek(long fpos)
**              fpos        array with file positions of packet frames
**                          in dumpfile
** DESCRIPTION: positions the file pointer to the start of the packet frame
**              indicated by fpos.
** RETURNS:     void
** SEE ALSO:    
*************************************************************************/
void DpuDumpSeek(long fpos)
{
    fseek(dumpfh, fpos+FR_DUMP_HDRSIZE, SEEK_SET);
}

/************************************************************************
** NAME:        DpuDumpReadFrame
** SYNOPSIS:    void DpuDumpReadFrame(frp, buf)
**              DPBUF *frp          Pkt description struct
**              BYTE *buf           Pkt data buffer
** DESCRIPTION: Reads a packet from dumpfile.
** RETURNS:     void
** SEE ALSO:    DpuDumpInit, DumpEnd, DpuDumpWriteFrame
*************************************************************************/
void DpuDumpReadFrame(DPBUF *frp, BYTE *buf)
{
    if (dumpfh)
    {
        fread(frp, sizeof(DPBUF), 1, dumpfh);
        frp->pBuf = buf;
        fread(buf, sizeof(BYTE), frp->Size, dumpfh);
    }
}

/************************************************************************
** NAME:        DpuDumpWriteFrame
** SYNOPSIS:    void DpuDumpWriteFrame(frp)
**              DPBUF *frp          Pkt description struct
** DESCRIPTION: Writes a packet to dumpfile.
** RETURNS:     void
** SEE ALSO:    DpuDumpInit, DumpEnd, DpuDumpReadFrame
*************************************************************************/
void DpuDumpWriteFrame(DPBUF *frp)
{
    unsigned int len;

    if (dumpfh)
    {
        fputc(REC_FRAME, dumpfh);
        len = sizeof(DPBUF)+frp->Size + FR_DUMP_HDRSIZE;
        fwrite(&len, sizeof(len), 1, dumpfh);
        fwrite(frp, sizeof(DPBUF), 1, dumpfh);
        fwrite(frp->pBuf, sizeof(BYTE), frp->Size, dumpfh);
    }
}

/************************************************************************
** NAME:        DpuDumpEnd
** SYNOPSIS:    void DpuDumpEnd(void)
** DESCRIPTION: Closes the packet to dumpfile.
** RETURNS:     void
** SEE ALSO:    DpuDumpInit, DpuDumpReadFrame, DpuDumpWriteFrame
*************************************************************************/
void DpuDumpEnd(void)
{
    if (dumpfh)
    {
        fclose(dumpfh);
        dumpfh = (DPBUF *)0;
    }
}
