/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       IPSEND.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/6/11
**
** DESCRIPTION: Contains the send function of the IP layer.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   IPSEND.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPSEND.C_V  $
**              
**                 Rev 1.1   21 Nov 1990 14:24:24   etstjan
**              No explicit note
**              
**                 Rev 1.0   20 Nov 1990 16:15:06   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/IPSEND.C_V   1.1   21 Nov 1990 14:24:24   etstjan  $";
#endif

#include      <stdio.h>
#include      <stdlib.h>
#include      <string.h>
#include      <power3.h>
#include      <beholder.h>

#include      "ip.h"                   /* general ip defines */
#include      "ipcodes.h"              /* and ip return codes */
#include      "iplib.h"                /* include ip library */
#include      "ipif.h"                 /* interface layer */
#include      "ipevents.h"             /* event processing */
#include      "iplayer.h"

/* prototypes */
static int SendPacket(CIPHEAD *ThisHeader, DATALINK *Packet,
                      int BCast, IFDESCRIPTOR *FirstHopIf,
                      ADDRESS_T FirstHop);


/**************************************************************
** NAME:        IPSend
** SYNOPSIS:    int IPSend(CIPHEAD *ThisHeader,
**                         DATALINK *ThisPacket,
**                         DIR_CHOICE Direct);
**
** DESCRIPTION: Sends an IP packet through an interface.
**              ThisHeader should have the following fields
**              filled: Tos, Ident, Flags (except 'more
**              fragments'), Time, Protocol, Source,
**              Destination and OLength/OpState if ThisHeader
**              contains options. Other options may be
**              passed in a datalink with the type
**              'OPTIONSHD'.
**              The routing table is expected to find the
**              first hop Internet Address. When 'Direct'
**              is set to 'DIRECT_ONLY', no intermediate
**              hops will be allowed.
**              ThisPacket must contain the data to be sent.
**              If the packet to be sent is longer then
**              the network's Maximum Message Size, the packet
**              is fragmented.
**              Broadcast packets will be sent through all
**              interfaces with Internet Addresses matching
**              the broadcast address. 
**     
** RETURNS:     NO_ERR      -->   no error
**              LENGTH_ERR  -->   too many options in packet
**              else        -->   error code (see IPCODES.H)
**************************************************************/
int IPSend(CIPHEAD *ThisHeader, DATALINK *ThisPacket, DIR_CHOICE Direct)
{
  DATALINK      *TmpPnt;
  DATALINK      **PrevPnt = &ThisPacket;
  ADDRESS_T     FirstHop;
  IFDESCRIPTOR  *FirstHopIf;
  int           RetCode = NO_ERR;
  int           TempCode;
  int           BCast;                 /* broadcast? */

  ThisHeader->Version = VERSION;
  ThisHeader->HLength = 20;
  ThisHeader->Flag &= ~MORE_FRAGMENT;  /* only one fragment */
  ThisHeader->Offset = 0;

  /* search datalink for options and copy them to header */
  while (((TmpPnt = *PrevPnt) != NULL) && (RetCode == NO_ERR)) {
    if (TmpPnt->ThisType == OPTIONSHD) {
      RetCode = OptionsToCip(ThisHeader, TmpPnt->DataStart, TmpPnt->Length);
      *PrevPnt = TmpPnt->NextOne;
      IPDataLinkFree(TmpPnt);
      }
    else {
      PrevPnt = &(TmpPnt->NextOne);
      }
    }
  if (RetCode != NO_ERR) {             /* an error, so quit */
    if (RetCode == LENGTHERR) {
      IPStat.IllSndOption++;
      }
    IPReport(RetCode, NULL, ThisPacket);
    return RetCode;
    }

  ThisHeader->TLength = ThisHeader->HLength + DLinkLen(ThisPacket);
  AllignOptions(ThisHeader);

  if (ThisHeader->OLength > 0) {
    if (ThisHeader->OpState & STRICT_SRC) 
      Direct = DIRECT_ONLY;
    else 
      if (ThisHeader->OpState & LOOSE_SRC) 
        Direct = HOPS_ALLOWED;
    }

  if ((RetCode = GetFirstHop(&FirstHopIf, &FirstHop,
                             ThisHeader->Destin, ThisHeader->Tos,
                             &BCast, Direct)) != NO_ERR) {
    IPReport(RetCode, NULL, ThisPacket);
    return RetCode;
    }
  if (BCast) {
    IPStat.NrBCastS++;
    }

  while (FirstHopIf != NULL) {
    IPStat.NrPackSend++;
    TempCode = SendPacket(ThisHeader, ThisPacket, BCast,
                          FirstHopIf, FirstHop); 
    if (RetCode == NO_ERR) RetCode = TempCode;
    FirstHopIf = (BCast) ? FirstHopIf->BCastLink : NULL;
    }

  DiscardAll(ThisPacket);
  return RetCode;
}

/**************************************************************
** NAME:        SendPacket
** SYNOPSIS:    static int SendPacket(CIPHEAD *ThisHeader,
**                                    DATALINK *Packet,
**                                    int BCast,
**                                    IFDESCRIPTOR *FirstHopIf,
**                                    ADDRESS_T FirstHop)
**
** DESCRIPTION: Sends 'Packet' through interface 'FirstHopIf'.
**              'ThisHeader' is converted to a byte stream
**              and prepended to the packet.
**              If necessary, the packet is fragmented. In
**              this case, the 'non copy options' are removed
**              from the IP header.
**              The datalink 'Packet' is NOT freed in this
**              function.
** RETURNS:     NO_ERR       -->   no error
**              FRAG_NOT_ALL -->   fragmentation necessary &
**                                 'Don't Fragment' flag set
**              else         -->   error code (see IPCODES.H) 
**************************************************************/
static int SendPacket(CIPHEAD *ThisHeader, DATALINK *Packet,
                      int BCast, IFDESCRIPTOR *FirstHopIf,
                      ADDRESS_T FirstHop)
{
  CIPHEAD       FragHeader;
  USHORT        MaxSndSz;
  LLREQ         KindReq = (BCast) ? BROADREQ : SENDREQ;
  DATALINK      *RstPnt;
  DATALINK      *HeadLink;
  DATALINK      *SaveLast;
  int           RetCode;
  int           NrFragBlocks;          /* # of 8 byte blocks in fragment */
  USHORT        RestLength;
  USHORT        CurOffset;             /* offset of current fragment */

  if ((HeadLink = IPDataLinkGet(ThisHeader->HLength)) == NULL) {
    IPReport(NOSPACE, NULL, NULL);
    return NOSPACE; 
    }

  /* if sent to gateway, maximum packet size is MINMMS */
  MaxSndSz = (FirstHop == ThisHeader->Destin) ? FirstHopIf->Mms_S : MINMMS;
  HeadLink->ThisType = IPHD;
  RetCode = CipToByte(HeadLink->DataStart, ThisHeader);
  if (RetCode != NO_ERR) {
    IPStat.IllHeader++;
    IPReport(RetCode, NULL, NULL);
    IPDataLinkFree(HeadLink);
    return RetCode;
    }
  HeadLink->NextOne = Packet;          /* extend datalink */

  if (ThisHeader->TLength <= MaxSndSz) {
    RetCode = (*(FirstHopIf->SendFun))(KindReq, HeadLink, FirstHop);
    }
  else {          /* fragmentation necessary */
    if (((ThisHeader->Flag & DONT_FRAGMENT) != 0) ||
        ((USHORT)(ThisHeader->HLength) + 8u > MaxSndSz)) {
      IPDataLinkFree(HeadLink);
      return (ThisHeader->Flag & DONT_FRAGMENT) ? FRAG_NOT_ALL : MMS_TOO_SMALL;
      }
    FragHeader = *ThisHeader;
    NrFragBlocks = (MaxSndSz - FragHeader.HLength) / 8;

    if ((RstPnt = SplitDataLink(Packet, &SaveLast,
                                NrFragBlocks * 8)) == NULL) {
      IPReport(NOSPACE, NULL, NULL);
      IPDataLinkFree(HeadLink);
      return NOSPACE;
      }

    RestLength = FragHeader.TLength - FragHeader.HLength - NrFragBlocks * 8;
    CurOffset = NrFragBlocks;
    FragHeader.Flag |= MORE_FRAGMENT;
    FragHeader.TLength = FragHeader.HLength + (NrFragBlocks * 8);
    CipToByte(HeadLink->DataStart, &FragHeader);

    /* send first fragment */
    IPStat.NrFragSend++;
    RetCode = (*(FirstHopIf->SendFun))(KindReq, HeadLink, FirstHop);
    SaveLast->NextOne = RstPnt;        /* restore datalink to old state */
    if (RetCode != NO_ERR) {
      IPDataLinkFree(HeadLink);
      return RetCode;
      }

    /* prepare other fragments */
    RemoveNonCopyOptions(&FragHeader);
    NrFragBlocks = (MaxSndSz - FragHeader.HLength) / 8;
    FragHeader.TLength = FragHeader.HLength + (NrFragBlocks * 8);

    while ((RestLength + FragHeader.HLength) > MaxSndSz) {
      /* further fragmentation necessary */
      HeadLink->NextOne = RstPnt;
      if ((RstPnt = SplitDataLink(RstPnt, &SaveLast,
                                  NrFragBlocks * 8)) == NULL) {
        IPReport(NOSPACE, NULL, NULL);
        IPDataLinkFree(HeadLink);
        return NOSPACE;
        }

      FragHeader.Offset = CurOffset;
      CurOffset += NrFragBlocks;
      HeadLink->Length = FragHeader.HLength;
      CipToByte(HeadLink->DataStart, &FragHeader);

      /* send next fragment */
      IPStat.NrFragSend++;
      RetCode = (*(FirstHopIf->SendFun))(KindReq, HeadLink, FirstHop);
      SaveLast->NextOne = RstPnt;   /* restore datalink */
        if (RetCode != NO_ERR) {
        IPDataLinkFree(HeadLink);
        return RetCode;
        }

      RestLength -= NrFragBlocks * 8;
      }

    /* send last fragment */
    IPStat.NrFragSend++;
    FragHeader.Offset = CurOffset;
    FragHeader.Flag &= ~MORE_FRAGMENT;
    FragHeader.TLength = RestLength + FragHeader.HLength;
    HeadLink->Length = FragHeader.HLength;
    HeadLink->NextOne = RstPnt;

    CipToByte(HeadLink->DataStart, &FragHeader);

    RetCode = (*(FirstHopIf->SendFun))(KindReq, HeadLink, FirstHop);
    SaveLast->NextOne = RstPnt;     /* restore datalink */
    }

  IPDataLinkFree(HeadLink);
  return RetCode;
}



